// V1.0 17th July 2007
//
// Copyright 2007 Phil Harman VK6APH
// Copyright 2008 Scotty Cowling WA2DFI
//
//  HPSDR - High Performance Software Defined Radio
//
//
//  ADC module for driving ADC78H90 
//
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA// SCI inteface to ADC78H90 

//  Penelope uses inputs AIN1 to AIN5
//  SCLK can run at 8MHz max and is = clock/4 
//  Note: AINx data can be latched using nADCCS

//  for now just use AIN5 the Forward Power sensor value
//
//  22 Apr 2008	SC	Added functionality to read AIN1-AIN4 to support production testing
//

module ADC(clock, SCLK, nADCCS, MISO, MOSI, AIN1, AIN2, AIN3, AIN4, AIN5);

// clock input to module
input  	clock;

// ADC interface
input  	MISO;			// ADC DOUT pin
output 	MOSI;			// ADC DIN pin
output  nADCCS;			// ADC nCS pin
output 	SCLK;			// ADC SCLK pin

// Analog input data holding registers
output 	[11:0]AIN1;		// Analog input 1
output 	[11:0]AIN2;		// Analog input 2
output 	[11:0]AIN3;		// Analog input 3
output 	[11:0]AIN4;		// Analog input 4
output 	[11:0]AIN5;		// Analog input 5





// ADC interface 
reg  MOSI;				// ADC DIN pin
reg  nADCCS = 1'b1;		// ADC nCS pin
reg  SCLK = 1'b1;		// ADC SCLK pin

// Analog input data holding registers
reg  [11:0] AIN1; 		// reg holds AIN1 value
reg  [11:0] AIN2; 		// reg holds AIN2 value
reg  [11:0] AIN3; 		// reg holds AIN3 value
reg  [11:0] AIN4; 		// reg holds AIN4 value
reg  [11:0] AIN5; 		// reg holds VFWD volts

// Temporary data holding registers
reg  [11:0] temp_AIN1;	// temp for AIN1
reg  [11:0] temp_AIN2;	// temp for AIN2
reg  [11:0] temp_AIN3;	// temp for AIN3
reg  [11:0] temp_AIN4;	// temp for AIN4
reg  [11:0] temp_AIN5;	// temp for AIN5

// ADC input data to select Analog channel
reg [15:0] data;		// ADC shift in address

// here are the address values made from chan[2:0], just for examples
//assign data1 = 16'b0000_0000_0000_0000; // address of AIN1
//assign data2 = 16'b0000_1000_0000_0000; // address of AIN2
//assign data3 = 16'b0001_0000_0000_0000; // address of AIN3
//assign data4 = 16'b0001_1000_0000_0000; // address of AIN4
//assign data5 = 16'b0010_0000_0000_0000; // address of AIN5

// State machine counters
reg  [2:0]ADC;			// state
reg  [3:0]bit_cnt;		// bit counter
reg	 [2:0]chan;			// channel number 0-4 maps to AIN1-AIN5

// NOTE: this code generates the SCLK clock for the ADC
//
always @ (posedge clock)
	begin
		case (ADC)

		0:	begin
			nADCCS <= 1'b1; 			// set nCS high
			bit_cnt <= 4'd15; 			// set bit counter
			chan <= 3'd0;
			data <= 16'd0;
			ADC <= 1;
			end
	
		1:  begin
			nADCCS <= 0; 				// select ADC
			MOSI <= data[bit_cnt];		// shift data out to ADC DIN pin
			ADC <= 2;
			end
	
		2:  begin
			SCLK <= 0;					// SCLK low
			ADC <= 3;
			end
	
		3:  begin
			ADC <= 4;
			end

		4:  begin
				SCLK 	<= 1;				// SCLK high
				ADC 	<= 1;
				data 	<= {2'b0,chan[2:0],11'b0};
				if (bit_cnt == 0)begin 	// next word - change channel address here
					bit_cnt <= 4'd15;
					if (chan == 3'h4) begin		// count channels 0 - 4
						chan <= 3'h0;
					end
					else begin
						chan <= chan + 3'h1;
					end
				end
				else begin
					bit_cnt <= bit_cnt - 1'b1; // do all again
				end
			end 
	
		default: begin
				ADC <= 0;
				chan <= 0;
			end
		endcase
	end 


// SPI receiver - doesn't get much simpler than this!
// Note that the value read is one behind due to pipelining in the ADC
//


always @ (posedge clock)
	if (ADC == 3'b100) begin
			case (chan)
			0:	begin
					if (bit_cnt > 11) begin
						AIN5 <= temp_AIN5;
					end
					else begin
						temp_AIN5[bit_cnt] <= MISO; 	// capture incoming data, chan 5
					end
				end

			1:	begin
					if (bit_cnt > 11) begin
						AIN1 <= temp_AIN1;
					end
					else begin
						temp_AIN1[bit_cnt] <= MISO; 	// capture incoming data, chan 1
					end
				end

			2:	begin
					if (bit_cnt > 11) begin
						AIN2 <= temp_AIN2;
					end
					else begin
						temp_AIN2[bit_cnt] <= MISO; 	// capture incoming data, chan 2
					end
				end

			3:	begin
					if (bit_cnt > 11) begin
						AIN3 <= temp_AIN3;
					end
					else begin
						temp_AIN3[bit_cnt] <= MISO; 	// capture incoming data, chan 3
					end
				end

			4:	begin
					if (bit_cnt > 11) begin
						AIN4 <= temp_AIN4;
					end
					else begin
						temp_AIN4[bit_cnt] <= MISO; 	// capture incoming data, chan 4
					end
				end

			default: begin								// dummy

				end
				

			endcase
	end 



endmodule 

