/***********************************************************
*
*	Mercury DDC Receiver Board Production Test 
*
************************************************************/

// V1.0 4 December 2008 

// (C) Scott Cowling WA2DFI 2008
// (C) Phil Harman VK6APH 2006, 2007, 2008


//  HPSDR - High Performance Software Defined Radio
//
//  Mercury Test Program - DO NOT USE WITH ANYTHING OTHER THAN LOOPBACK CABLE
//
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


/* 	This program performs a loop-back test on the Mercury Board.
	It requires loopback plugs on J1, J5 and J6. 
	Pass is indicated by LED D4 on and rotating pattern on LEDs D5-D11.
	
	Revision log:
	
	4 December 2008	SC	Initial Version 
	10 December 2008 SC	Final version 
*/

//////////////////////////////////////////////////////////////
//
//              Quartus V7.2 SP2 Notes
//
//////////////////////////////////////////////////////////////

/*
	In order to get this code to compile without timing errors under
	Quartus V7.2 SP2 I needed to use the following settings:
	
	- Timing Analysis Settings - Use Classic 
	- Analysis and Synthesis Settings\Power Up Dont Care [not checked]
	- Analysis and Synthesis Settings\Restructure Multiplexers  [OFF]
	- Fitter Settings\Optimise fast-corner timing [ON]
	
*/

	
module Mercury_Prod_Test(

// (2) clock inputs
	input   OSC_10MHZ,		// 10MHz TCXO input (pin 34) 
	input   CLKA,			// 122.88MHz clock from LT2208 (pin 209)

// (5) CODEC clock and data
	output CDIN,			// Rx audio out to TLV320 (pin 142)
	output CBCLK, 			// 3.072MHz BCLK from Atlas C8 (pin 139)
	output CLRCLK, 			// 48KHz L/R clock from Atlas C9 (pin 143)
	output CLRCOUT,			// (pin 144)
	output CMCLK, 			// 12.288MHz master clock from Atlas C17 (pin 132)

// (4) CODEC SPI outputs
	output MOSI,			// SPI interface to TLV320 (pin 134)
	output SCLK,			// SPI interface to TLV320 (pin 133)
	output nCS,				// SPI interface to TLV320 (pin 137)
	output CMODE,			// SPI interface to TLV320 (pin 135)

// (4) Alex SPI inputs/outputs
	output SPI_data,		// SPI data to Alex (pin 37)
	input SPI_clock,		// SPI clock to Alex (pin 38)
	input Tx_load_strobe,	// SPI Tx data load strobe to Alex (pin 41)
	output Rx_load_strobe,	// SPI Rx data load strobe to Alex (pin 39)

// (3) clock control outputs
	output FPGA_PLL, 		// PLL control volts to loop filter (pin 240)
	output LVDS_TXE,		// LVDS Tx enable (pin 238)
	output LVDS_RXE_N,		// LVDS Rx enable (pin 239)

// (21) ADC data and control
	input [15:0]INA,		// samples from LT2208 (pins 184-189, 194-197, 200-203, 207, 214)
	input  OVERFLOW,		// ADC overflow bit (pin 216)
	output DITHER,			// ADC dither control bit (pin 217)
	output SHDN,			// ADC shutdown bit (pin 223)
	output PGA, 			// ADC preamp gain (pin 224)
	output RAND, 			// ADC ramdonizer bit (pin 230)

// (4) Test outputs
	output TEST0,			// Test point (pin 43)
	output TEST1,			// Test point (pin 44)
	output TEST2,			// Test point (pin 45)
	output TEST3,			// Test point (pin 46)

// (8) LED outputs, active high to light
	output DEBUG_LED0,		// Debug LED0 (D11, pin 22)
	output DEBUG_LED1,		// Debug LED1 (D10, pin 21)
	output DEBUG_LED2,		// Debug LED2 (D9, pin 18)
	output DEBUG_LED3,		// Debug LED3 (D8, pin 13)
	output DEBUG_LED4,		// Debug LED4 (D7, pin 9)
	output DEBUG_LED5,		// Debug LED5 (D6, pin 6)
	output DEBUG_LED6,		// Debug LED6 (D5, pin 5)
	output DEBUG_LED7,		// Debug LED7 (D4, pin 4)

// (1) control outputs
	output ATT_ON,			// high to turn ATT relay on (pin 183)

// (22) Atlas bus pins, A row
	output A2,			// (pin 122)
	output A3,			// (pin 126)
	output A4,			// (pin 119)
	output A5,			// (pin 117)
	output A6,			// (pin 113)
	output A7,			// (pin 111)
	output A8,			// (pin 109)
	output A9,			// (pin 107)
	output A10,			// (pin 103)
	output A11,			// (pin 99)
	output A12,			// (pin 95)
	output A13,			// (pin 93)
	output A14,			// (pin 87)
	output A15,			// (pin 83)
	output A16,			// (pin 81)
	output A17,			// (pin 78)
	output A18,			// (pin 73)
	output A19,			// (pin 71)
	output A20,  		// (pin 69)
	output A21,  		// (pin 65)
	output A22,			// (pin 63)
	output A31,			// (pin 49)

// (27) Atlas bus pins, C row
	input C2,			// (pin 131)
	input C3,			// (pin 127)
	input C4,			// (pin 120)
	input C5,			// (pin 118)
	input C6,			// (pin 114)
	input C7,			// (pin 112)
	input C8,			// (pin 110)
	input C9,			// (pin 108)
	input C10,			// (pin 106)
	input C11,			// (pin 100)
	input C12,			// (pin 98)
	input C13,			// (pin 94)
	input C14,			// (pin 88)
	input C15,			// (pin 84)
	input C16,			// (pin 82)
	input C17,			// (pin 80)
	input C18,			// (pin 76)
	input C19,			// (pin 72)
	input C20,			// (pin 70)
	input C21,			// (pin 68)
	input C22,			// (pin 64)
	input C23,			// (pin 57)
	input C24,			// (pin 56)
	input C25,			// (pin 55)
	input C27,			// (pin 52)
	input C29,			// (pin 51)
	input C31,			// (pin 50)

// (8) GPIO output pins
	output GPIO2,		// (pin 146)
	output GPIO4,		// (pin 148)
	output GPIO6,		// (pin 161)
	output GPIO8,		// (pin 166)
	output GPIO10,		// (pin 168)
	output GPIO12,		// (pin 171)
	output GPIO14,		// (pin 176)
	output GPIO16,		// (pin 181)

// (9) GPIO input pins
	input GPIO1,		// (pin 145)
	input GPIO3,		// (pin 147)
	input GPIO5,		// (pin 160)
	input GPIO7,		// (pin 162)
	input GPIO9,		// (pin 167)
	input GPIO11,		// (pin 169)
	input GPIO13,		// (pin 173)
	input GPIO15,		// (pin 177)
	input GPIO17,		// (pin 182)

// (2) FPGA control
	input n_RST,		// FPGA reset input (pin 236)
	output INIT_DONE	// INIT_DONE LED (pin 159)

);

/*
Test matrix

loopback				Merc		Merc		sch				sch
result			test	conn		conn		signal			signal
bit				nr		output		input		output			input
--------		----	-------		-------		---------		---------
0				0x00	P1-A29		P1-C29		TDI	(PD)		ATLAS_C29
1				0x01	P1-A27		P1-C27		TDO (PD)		ATLAS_C27

2				0x02	P1-A24		P1-C24		TCK	(PD)		ATLAS_C24
3				0x03	P1-A23		P1-C23		TMS	(PD)		ATLAS_C23

4				0x04	P1-A31		P1-C31		ATLAS_A31		ATLAS_C31

5				0x05	P1-A22		P1-C22		ATLAS_A22		ATLAS_C22
6				0x06	P1-A21		P1-C21		ATLAS_A21		ATLAS_C21
7				0x07	P1-A20		P1-C20		ATLAS_A20		ATLAS_C20
8				0x08	P1-A19		P1-C19		ATLAS_A19		ATLAS_C19
9				0x09	P1-A18		P1-C18		ATLAS_A18		ATLAS_C18
10				0x0a	P1-A17		P1-C17		ATLAS_A17		ATLAS_C17
11				0x0b	P1-A16		P1-C16		ATLAS_A16		ATLAS_C16
12				0x0c	P1-A15		P1-C15		ATLAS_A15		ATLAS_C15
13				0x0d	P1-A14		P1-C14		ATLAS_A14		ATLAS_C14
14				0x0e	P1-A13		P1-C13		ATLAS_A13		ATLAS_C13
15				0x0f	P1-A12		P1-C12		ATLAS_A12		ATLAS_C12
16				0x10	P1-A11		P1-C11		ATLAS_A11		ATLAS_C11
17				0x11	P1-A10		P1-C10		ATLAS_A10		ATLAS_C10
18				0x12	P1-A09		P1-C09		ATLAS_A9		ATLAS_C9
19				0x13	P1-A08		P1-C08		ATLAS_A8		ATLAS_C8
20				0x14	P1-A07		P1-C07		ATLAS_A7		ATLAS_C7
21				0x15	P1-A06		P1-C06		ATLAS_A6		ATLAS_C6
22				0x16	P1-A05		P1-C05		ATLAS_A5		ATLAS_C5
23				0x17	P1-A04		P1-C04		ATLAS_A4		ATLAS_C4
24				0x18	P1-A03		P1-C03		ATLAS_A3		ATLAS_C3
25				0x19	P1-A02		P1-C02		ATLAS_A2		ATLAS_C2

26				0x1a	J5-4		J5-3		FPGA_GPIO16		FPGA_GPIO17
27				0x1b	J5-6		J5-5		FPGA_GPIO14		FPGA_GPIO15
28				0x1c	J5-8		J5-7		FPGA_GPIO12		FPGA_GPIO13
29				0x1d	J5-10		J5-9		FPGA_GPIO10		FPGA_GPIO11
30				0x1e	J5-12		J5-11		FPGA_GPIO8		FPGA_GPIO9
31				0x1f	J5-14		J5-13		FPGA_GPIO6		FPGA_GPIO7
32				0x20	J5-16		J5-15		FPGA_GPIO4		FPGA_GPIO5
33				0x21	J5-18		J5-17,19	FPGA_GPIO2		FPGA_GPIO3, FPGA_GPIO1


34				0x22	J6-3		J6-4		SPI_data		SPI_clock
35				0x23	J6-5		J6-6		Rx_load_strobe	Tx_load_strobe

*/


////////////////////////////////////////////////////////////////////////
//
//		Initialize unused outputs
//
////////////////////////////////////////////////////////////////////////


// turn off CODEC
//
assign	CDIN =		1'b0;	// Rx audio out to TLV320 (pin 142)
assign	CBCLK =		1'b0;	// 3.072MHz BCLK from Atlas C8 (pin 139)
assign	CLRCLK =	1'b0;	// 48KHz L/R clock from Atlas C9 (pin 143)
assign	CLRCOUT =	1'b0;	// (pin 144)
assign	CMCLK =		1'b0;	// 12.288MHz master clock from Atlas C17 (pin 132)

// turn off CODEC SPI
//
assign	MOSI =	1'b0;		// SPI interface to TLV320 (pin 134)
assign	SCLK =	1'b0;		// SPI interface to TLV320 (pin 133)
assign	nCS =	1'b1;		// SPI interface to TLV320 (pin 137)
assign	CMODE =	1'b0;		// SPI interface to TLV320 (pin 135)

// Turn off ADC
//
assign	DITHER =	1'b0;		// turn off dither
assign	SHDN =	1'b0;		// 0 = enable ADC so we can get HS clock
assign	PGA =	1'b0;		// turn amp
assign	RAND =	1'b0;		// turn off rand

// turn on INIT_DONE LED (D3)
//
assign INIT_DONE = 1'b0;	// turn INIT_DONE LED on

// ATT_ON not used
//
assign	ATT_ON = 1'b0;		// turn off relay 


////////////////////////////////////////////////////////////////////////
//
//		Clocks and clock drive enables
//
////////////////////////////////////////////////////////////////////////

// externally loop-back the 122.88MHz clock through LVDS driver/receiver
//
assign	LVDS_TXE =	 1'b1;		// 1= drive 122.88MHz off board (LVDS Tx enable, pin 238)
assign	LVDS_RXE_N = 1'b0;		// 0= receive external 122.88MHz (LVDS Rx enable pin 239)

									
///////////////////////////////////////////////////////////
//
//    122.88MHz PLL and clock control 
//
///////////////////////////////////////////////////////////

/* 
	Divide the 10MHz reference and 122.88MHz clock to give 80kHz signals.
	Apply these to an EXOR phase detector. If the 10MHz reference is not
	present the EXOR output will be a 80kHz square wave. When passed through 
	the loop filter this will provide a dc level of (3.3/2)v which will
	set the 122.88MHz VCXO to its nominal frequency.
	The selection of the internal or external 10MHz reference for the PLL
	is made using ref_ext.
*/

// div 10 MHz ref clock by 125 to get 80 khz 

wire ref_80khz; 
reg osc_80khz; 

oddClockDivider refClockDivider(OSC_10MHZ, ref_80khz); 

// Divide  122.88 MHz by 1536 to get 80 khz 
reg [9:0] count_12288; 

always @ (posedge CLKA) begin
	if (count_12288 == 767) begin
		count_12288	<= 0;
		osc_80khz	<= ~osc_80khz; 
	end
    else begin
		count_12288 <= 1'b1 + count_12288;
	end
end

// NOTE: If 10MHz reference is not available then this lock detector
// will show "in lock". However, the rest of the loopback test will
// fail because it is clocked on the 10MHz clock.
//
// Apply to EXOR phase detector 
assign FPGA_PLL = ref_80khz ^ osc_80khz; 

// Light LED if duty cycle on FPGA_PLL is too large or too small
//
reg	[10:0] lock_count;		// max count 11 bits or 2047
reg	FPGA_PLL_Q;				// delay for edge detection
reg	PLL_LOCK;				// 1=PLL in lock

always @ (posedge CLKA) begin
	FPGA_PLL_Q	<= FPGA_PLL;
	if (FPGA_PLL) begin
		if (!FPGA_PLL_Q) begin
			lock_count	<= 11'h0;
		end
		else if (lock_count != 11'h3FF) begin
			lock_count	<= lock_count + 11'h001;
		end
	end
	else if ( (lock_count[10:7] >= 4'b0001) && (lock_count[10:7] <= 4'b1011) ) begin
		PLL_LOCK 	<= 1'b1;	// in lock
	end
	else begin
		PLL_LOCK	<= 1'b0;	// out of lock
	end
end

// Generate heartbeat timing from 122.88MHz
//
reg [27:0] clock_reg;

always @ ( posedge  CLKA ) begin
   clock_reg <= clock_reg + 1'b1;
end


// Test pins
assign TEST0 = osc_80khz; 	// 80kHz from 122.88MHz
assign TEST1 = ref_80khz; 	// 80kHz from 10MHz
assign TEST2 = FPGA_PLL;  	// phase detector output
assign TEST3 = PLL_LOCK; 	// 1=PLL in lock

//////////////////////////////////////////////////////////////
//
//		LED Rotation Generator 
//
//////////////////////////////////////////////////////////////

//
// Seven LEDs rotate unless an error is detected in the loopback result
// register. When error is detected, they hold all on.
// The eighth LED lights when the 122.88MHz input is phase-locked
// to the 10MHz clock, and is out if 122.88MHz is missing or out of lock.

reg [24:0] clock_count;
reg [7:0] leds;
assign DEBUG_LED7 = leds[7];
assign DEBUG_LED6 = leds[6];
assign DEBUG_LED5 = leds[5];
assign DEBUG_LED4 = leds[4];
assign DEBUG_LED3 = leds[3];
assign DEBUG_LED2 = leds[2];
assign DEBUG_LED1 = leds[1];
assign DEBUG_LED0 = clock_reg[25] && PLL_LOCK; // heartbeat when PLL in lock, else solid on or off

always @ (posedge OSC_10MHZ) begin 
	if ( clock_count == 25'd1200000 ) begin
		clock_count <= 25'd0;
		if (|loopback_result[35:4]) begin

// failure lights LEDs iin groups
//
			leds[7] <= |loopback_result[35:34]; 	// Bits [35:34]
			leds[6] <= |loopback_result[33:30]; 	// Bits [33:30]
			leds[5] <= |loopback_result[29:26]; 	// Bits [29:26]
			leds[4] <= |loopback_result[25:19]; 	// Bits [25:19]
			leds[3] <= |loopback_result[18:12]; 	// Bits [18:12]
			leds[2] <= |loopback_result[11:5];	 	// Bits [11:5]
			leds[1] <= loopback_result[4]; 			// Bit [4]
		end

		else if ( (DEBUG_LED7 == 1'b1) || (leds[7:1] == 7'b0) ) begin 		// no error, msb high or all bits zero so set init pattern
			leds[7:1] 	<= 7'b00000001; 
		end

		else begin 								// no error, msb clear so shift
			leds[7:1]	<= leds[7:1] << 1;
		end
	end
			
	else begin 
		clock_count <= clock_count + 25'h1; 
	end
end 



//////////////////////////////////////////////////////////////
//
//		Loopback test 
//
//////////////////////////////////////////////////////////////

//
// Seven LEDs rotate unless an error is detected in the loopback result
// register. When error is detected, they hold all on.
// The eighth LED lights when the 122.88MHz input is phase-locked
// to the 10MHz clock, and is out if 122.88MHz is missing or out of lock.



// Assign stimulus output vector bits to output pins
//
//assign TDI = atlas_vec[0];	// not used, HI
//assign TDO = atlas_vec[1];	// not used, LO
//assign TCK = atlas_vec[2];	// not used, LO
//assign TMS = atlas_vec[3];	// not used, HI
assign A31 = atlas_vec[4];
assign A22 = atlas_vec[5];
assign A21 = atlas_vec[6];
assign A20 = atlas_vec[7];
assign A19 = atlas_vec[8];
assign A18 = atlas_vec[9];
assign A17 = atlas_vec[10];
assign A16 = atlas_vec[11];
assign A15 = atlas_vec[12];
assign A14 = atlas_vec[13];
assign A13 = atlas_vec[14];
assign A12 = atlas_vec[15];
assign A11 = atlas_vec[16];
assign A10 = atlas_vec[17];
assign A9 = atlas_vec[18];
assign A8 = atlas_vec[19];
assign A7 = atlas_vec[20];
assign A6 = atlas_vec[21];
assign A5 = atlas_vec[22];
assign A4 = atlas_vec[23];
assign A3 = atlas_vec[24];
assign A2 = atlas_vec[25];
assign GPIO16 = atlas_vec[26];
assign GPIO14 = atlas_vec[27];
assign GPIO12 = atlas_vec[28];
assign GPIO10 = atlas_vec[29];
assign GPIO8 = atlas_vec[30];
assign GPIO6 = atlas_vec[31];
assign GPIO4 = atlas_vec[32];
assign GPIO2 = atlas_vec[33];
assign SPI_data = atlas_vec[34];
assign Rx_load_strobe = atlas_vec[35];

// Assign input pins to response vector
//
assign atlas_resp = {Tx_load_strobe,	// bit 35
					SPI_clock,			// bit 34
					GPIO3,				// bit 33
					GPIO5,				// bit 32
					GPIO7,				// bit 31
					GPIO9,				// bit 30
					GPIO11,				// bit 29
					GPIO13,				// bit 28
					GPIO15,				// bit 27
					GPIO17,				// bit 26
					C2,					// bit 25
					C3,					// bit 24
					C4,					// bit 23
					C5,					// bit 22
					C6,					// bit 21
					C7,					// bit 20
					C8,					// bit 19
					C9,					// bit 18
					C10,				// bit 17
					C11,				// bit 16
					C12,				// bit 15
					C13,				// bit 14
					C14,				// bit 13
					C15,				// bit 12
					C16,				// bit 11
					C17,				// bit 10
					C18,				// bit 9
					C19,				// bit 8
					C20,				// bit 7
					C21,				// bit 6
					C22,				// bit 5
					C31,				// bit 4
					C23,				// bit 3
					C24,				// bit 2
					C27,				// bit 1
					C29};				// bit 0
					

//-----------------------------------------------------------
//
// Generate test pattern on Atlas vectors.
//

// Vectors for stimulus outputs
//
reg [35:0]	atlas_vec;			// vector for atlas stimulus outputs

// Vectors for response inputs
//
wire [35:0]	atlas_resp;			// vector for atlas response inputs

// Loopback vectors
//
wire [35:0]	loopback_reset; 	// 1=reset loopback bit, 0=do nothing
reg [35:0]	loopback_result; 	// 1=error, 0=OK

// Loopback test signals
//
reg	[7:0] 	cmp_cnt;			// used to determine interval for pattern shifting

assign	loopback_reset = 36'h0;		// not used. clear all errors every 120ms

always @ (posedge OSC_10MHZ) begin
	if (clock_count == 25'h0) begin
		atlas_vec	<= 36'h555555555;	// clear both vectors
		cmp_cnt		<= 8'h0;			// clear delay count
		loopback_result	<= 36'h0;		// clear errors
	end
	else begin 
		if (cmp_cnt	== 8'h10) begin				// time to update loopback_result
			loopback_result	<= (({atlas_vec[35:4], 4'b1001} ^ atlas_resp) | loopback_result);
		end

		if (cmp_cnt == 8'h20) begin				// time to change stimulus
			cmp_cnt			<= 8'h0;			// and restart delay count
//			loopback_result	<= loopback_result & (~loopback_reset);		// clear errors if required		
			if (atlas_vec == 36'h0) begin
				atlas_vec	<= 36'h555555555;		// if rotation complete, start with initial pattern
			end
			else begin
				atlas_vec	<= atlas_vec << 1;	// shift left
			end
		end
		else begin
			cmp_cnt			<= cmp_cnt + 8'b1;				// increment count
		end
	end
end							

endmodule 



