/*
 * Spectrum and Network Analyzer
 * Copyright 2005-2007 Darrell Harmon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * CORDIC Rotator
 * 16 bit real input, 18 bit I/Q output, gain = 3.2934
 * 20 bit phase input, scale: pi radians = 2^19
 * SFDR appears to be > 105 dB
 * Can be used for sin/cos or vector rotation
 * Fully pipelined, latency of 17 cycles
 */

`timescale 1ns / 10ps

module cordic_16(in, iout, qout, ain, clk);
   input                 clk;
   input [19:0]          ain; // Angle 
   input  [15:0]   in; // Real input to CORDIC rotator
   output  [17:0]  iout, qout; // I/Q output
   reg  [17:0]     iout, qout;   
   reg  [18:0]     a0;
   reg  [17:0]     a1;
   reg  [17:0]     a2;
   reg  [16:0]     a3;
   reg  [15:0]     a4;
   reg  [14:0]     a5;
   reg  [13:0]     a6;
   reg  [12:0]     a7;
   reg  [11:0]     a8;
   reg  [10:0]     a9;
   reg  [9:0]      a10;
   reg  [8:0]      a11;
   reg  [7:0]      a12;
   reg  [6:0]      a13;
   reg  [5:0]      a14;
   reg  [4:0]      a15;
   reg             a16;
   reg  [19:0]     i1, i2, i3, i4, i5, i6, i7, i8, i9;
   reg  [19:0]     i10, i11, i12, i13, i14, i15, i16, i17;
   reg  [19:0]     q0, q1, q2, q3, q4, q5, q6, q7, q8, q9;
   reg  [19:0]     q10, q11, q12, q13, q14, q15, q16, q17;

      
   always @ (posedge clk)
     begin
        /* Prerotate: 90 degrees */
        a0[17:0]  <= ain[17:0];
        a0[18] <= ~ain[18];
        if(ain[19])
          q0 <= -{in[15],in,3'd0};     
        else
          q0 <= {in[15],in,3'd0};
        /* 45 degrees */
        a1[16:0] <= a0[16:0];
        a1[17] <= ~a0[17];
        q1 <= q0;
        if(a0[18])
          i1 <= q0;
        else
          i1 <= -q0;
        /* 26.56 degrees */
        if(a1[17])
          begin
             a2 <= a1 + 18'd77376;
             i2 <= i1 + {q1[19], q1[19:1]};
             q2 <= q1 - {i1[19], i1[19:1]};
          end
        else
          begin
             a2 <= a1 - 18'd77376;
             i2 <= i1 - {q1[19], q1[19:1]};
             q2 <= q1 + {i1[19], i1[19:1]};
          end // else: !if(a1[17])
        /* 14.036 degrees */
        if(a2[17])
          begin
             a3 <= a2[16:0] + 17'd40884;
             i3 <= i2 + {{2{q2[19]}}, q2[19:2]};
             q3 <= q2 - {{2{i2[19]}}, i2[19:2]};
          end
        else
          begin
             a3 <= a2[16:0] - 17'd40884;
             i3 <= i2 - {{2{q2[19]}}, q2[19:2]};
             q3 <= q2 + {{2{i2[19]}}, i2[19:2]};
          end // else: !if(a2[17])
        /* 7.125 degrees */
        if(a3[16])
          begin
             a4 <= a3[15:0] + 16'd20753;
             i4 <= i3 + {{3{q3[19]}}, q3[19:3]};
             q4 <= q3 - {{3{i3[19]}}, i3[19:3]};
          end
        else
          begin
             a4 <= a3[15:0] - 16'd20753;
             i4 <= i3 - {{3{q3[19]}}, q3[19:3]};
             q4 <= q3 + {{3{i3[19]}}, i3[19:3]};
          end // else: !if(a3[16])
        /* 3.576 degrees */
        if(a4[15])
          begin
             a5 <= a4[14:0] + 15'd10417;
             i5 <= i4 + {{4{q4[19]}}, q4[19:4]};
             q5 <= q4 - {{4{i4[19]}}, i4[19:4]};
          end
        else
          begin
             a5 <= a4[14:0] - 15'd10417;
             i5 <= i4 - {{4{q4[19]}}, q4[19:4]};
             q5 <= q4 + {{4{i4[19]}}, i4[19:4]};
          end // else: !if(a4[15])
        /* 1.790 degrees */
        if(a5[14])
          begin
             a6 <= a5[13:0] + 14'd5213;
             i6 <= i5 + {{5{q5[19]}}, q5[19:5]};
             q6 <= q5 - {{5{i5[19]}}, i5[19:5]};
          end
        else
          begin
             a6 <= a5[13:0] - 14'd5213;
             i6 <= i5 - {{5{q5[19]}}, q5[19:5]};
             q6 <= q5 + {{5{i5[19]}}, i5[19:5]};
          end // else: !if(a5[14])
        /* 0.895 degrees */
        if(a6[13])
          begin
             a7 <= a6[12:0] + 13'd2607;
             i7 <= i6 + {{6{q6[19]}}, q6[19:6]};
             q7 <= q6 - {{6{i6[19]}}, i6[19:6]};
          end
        else
          begin
             a7 <= a6[12:0] - 13'd2607;
             i7 <= i6 - {{6{q6[19]}}, q6[19:6]};
             q7 <= q6 + {{6{i6[19]}}, i6[19:6]};
          end // else: !if(a6[14])
        /* 0.448 degrees */
        if(a7[12])
          begin
             a8 <= a7[11:0] + 12'd1304;
             i8 <= i7 + {{7{q7[19]}}, q7[19:7]};
             q8 <= q7 - {{7{i7[19]}}, i7[19:7]};
          end
        else
          begin
             a8 <= a7[11:0] - 12'd1304;
             i8<= i7 - {{7{q7[19]}}, q7[19:7]};
             q8 <= q7 + {{7{i7[19]}}, i7[19:7]};
          end // else: !if(a7[14])
        /* 0.224 degrees */
        if(a8[11])
          begin
             a9 <= a8[10:0] + 11'd652;
             i9 <= i8 + {{8{q8[19]}}, q8[19:8]};
             q9 <= q8 - {{8{i8[19]}}, i8[19:8]};
          end
        else
          begin
             a9 <= a8[10:0] - 11'd652;
             i9 <= i8 - {{8{q8[19]}}, q8[19:8]};
             q9 <= q8 + {{8{i8[19]}}, i8[19:8]};
          end // else: !if(a8[11])
        /* 0.1119 degrees */
        if(a9[10])
          begin
             a10 <= a9[9:0] + 10'd326;
             i10 <= i9 + {{9{q9[19]}}, q9[19:9]};
             q10 <= q9 - {{9{i9[19]}}, i9[19:9]};
          end
        else
          begin
             a10 <= a9[9:0] - 10'd326;
             i10 <= i9 - {{9{q9[19]}}, q9[19:9]};
             q10 <= q9 + {{9{i9[19]}}, i9[19:9]};
          end // else: !if(a9[10])
        /* 0.05595 degrees */
        if(a10[9])
          begin
             a11 <= a10[8:0] + 9'd163;
             i11 <= i10 + {{10{q10[19]}}, q10[19:10]};
             q11 <= q10 - {{10{i10[19]}}, i10[19:10]};
          end
        else
          begin
             a11 <= a10[8:0] - 9'd163;
             i11 <= i10 - {{10{q10[19]}}, q10[19:10]};
             q11 <= q10 + {{10{i10[19]}}, i10[19:10]};
          end // else: !if(a10[9])
        /* 0.02798 degrees */
        if(a11[8])
          begin
             a12 <= a11[7:0] + 8'd81;
             i12 <= i11 + {{11{q11[19]}}, q11[19:11]};
             q12 <= q11 - {{11{i11[19]}}, i11[19:11]};
          end
        else
          begin
             a12 <= a11[7:0] - 8'd81;
             i12 <= i11 - {{11{q11[19]}}, q11[19:11]};
             q12 <= q11 + {{11{i11[19]}}, i11[19:11]};
          end // else: !if(a11[8])
        /* 0.01399 degrees */
        if(a12[7])
          begin
             a13 <= a12[6:0] + 7'd41;
             i13 <= i12 + {{12{q12[19]}}, q12[19:12]};
             q13 <= q12 - {{12{i12[19]}}, i12[19:12]};
          end
        else
          begin
             a13 <= a12[6:0] - 7'd41;
             i13 <= i12 - {{12{q12[19]}}, q12[19:12]};
             q13 <= q12 + {{12{i12[19]}}, i12[19:12]};
          end // else: !if(a12[7])
        /* 0.00699 degrees */
        if(a13[6])
          begin 
             a14 <= a13[5:0] + 6'd20;
             i14 <= i13 + {{13{q13[19]}}, q13[19:13]};
             q14 <= q13 - {{13{i13[19]}}, i13[19:13]};
          end
        else
          begin
             a14 <= a13[5:0] - 6'd20;
             i14 <= i13 - {{13{q13[19]}}, q13[19:13]};
             q14 <= q13 + {{13{i13[19]}}, i13[19:13]};
          end // else: !if(a13[6])
        /* 0.00350 degrees */
        if(a14[5])
          begin
             a15 <= a14[4:0] + 5'd10;
             i15 <= i14 + {{14{q14[19]}}, q14[19:14]};
             q15 <= q14 - {{14{i14[19]}}, i14[19:14]};
          end
        else
          begin
             a15 <= a14[4:0] - 5'd10;
             i15 <= i14 - {{14{q14[19]}}, q14[19:14]};
             q15 <= q14 + {{14{i14[19]}}, i14[19:14]};
          end // else: !if(a14[5])

		iout <= i15[19:2];
		qout <= q15[19:2];

//        /* 0.00175 degrees */
//        if(a15[4])
//          begin
//             a16 <= (a15[3:0] < -4);
//             i16 <= i15 + {{15{q15[19]}}, q15[19:15]};
//             q16 <= q15 - {{15{i15[19]}}, i15[19:15]};
//          end
//        else
//          begin
//             a16 <= (a15[3:0] < 5);
//             i16 <= i15 - {{15{q15[19]}}, q15[19:15]};
//             q16 <= q15 + {{15{i15[19]}}, i15[19:15]};
//          end // else: !if(a15[4])
//        /* 0.00087 degrees */
//        if(a16)
//          begin
//             i17 <= i16 + {{16{q16[19]}}, q16[19:16]};
//             q17 <= q16 - {{16{i16[19]}}, i16[19:16]};
//          end
//        else
//          begin
//             i17 <= i16 - {{16{q16[19]}}, q16[19:16]};
//             q17 <= q16 + {{16{i16[19]}}, i16[19:16]};
//          end // else: !if(a16[3])
//        if(i17[1:0] > 2)
//          iout <= i17[19:2] + 18'd1;
//        else
//          iout <= i17[19:2];
//        if(q17[1:0] > 2)
//          qout <= q17[19:2] + 18'd1;
//        else
//          qout <= q17[19:2];


     end // always @ (posedge clk)
endmodule // cordic_16


