/*
 * Spectrum and Network Analyzer
 * Copyright 2005-2007 Darrell Harmon
 *
 * Updated by Cathy Moss (June 2008) to reduce rounding errors ..
 *  (which had deleted the DC output component) and added an extra 2-bit resolution all round.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * CORDIC Rotator
 * 16 bit real input, 18 bit I/Q output, gain = 3.2934
 * 22 bit phase input, scale: pi radians = 2^21
 * SFDR appears to be > 105 dB .. this value needs updating .. cathy
 * Can be used for sin/cos or vector rotation
 * Fully pipelined, latency of 19 cycles .. cathy doesn't know if it still is fully pipelined after her changes
 */



module cmcordic(in, iout, qout, ain, clk);

   input           clk;
   input   [21:0]  ain;			// Angle 
   input   [15:0]  in;			// Real input to CORDIC rotator
   output  [21:0]  iout, qout;	// I/Q output 
   
   reg  [21:0]     iout, qout;   
   
   reg  [20:0]     a0;
   reg  [19:0]     a1;
   reg  [19:0]     a2;
   reg  [18:0]     a3;
   reg  [17:0]     a4;
   reg  [16:0]     a5;
   reg  [15:0]     a6;
   reg  [14:0]     a7;
   reg  [13:0]     a8;
   reg  [12:0]     a9;
   reg  [11:0]     a10;
   reg  [10:0]     a11;
   reg  [9:0]      a12;
   reg  [8:0]      a13;
   reg  [7:0]      a14;
   reg  [6:0]      a15;
   reg  [5:0]      a16;
   reg  [4:0]      a17;
   reg  [3:0]      a18;
   
   reg  [21:0]     _in;
   
   reg  [21:0]     i1, i2, i3, i4, i5, i6, i7, i8, i9, i10, i11, i12, i13, i14, i15, i16, i17, i18, i19;
   reg  [21:0]     q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17, q18, q19;
   
   // the following registers are for minimizing rounding errors - not sure how to this without using extra registers - yet
   reg  [21:0]     i1b, i2b, i3b, i4b, i5b, i6b, i7b, i8b, i9b, i10b, i11b, i12b, i13b, i14b, i15b, i16b, i17b, i18b, i19b;
   reg  [21:0]     q1b, q2b, q3b, q4b, q5b, q6b, q7b, q8b, q9b, q10b, q11b, q12b, q13b, q14b, q15b, q16b, q17b, q18b, q19b;

  parameter a1_const  = 20'd309505; // 26.56505118 degrees = atan(1.0 / 2)
  parameter a2_const  = 21'd163534;	// 14.03624347 degrees = atan(1.0 / 4)
  parameter a3_const  = 18'd83012;	// 7.125016349 degrees = atan(1.0 / 8)
  parameter a4_const  = 17'd41667;	// 3.576334375 degrees = atan(1.0 / 16)
  parameter a5_const  = 16'd20854;	//
  parameter a6_const  = 15'd10430;	//
  parameter a7_const  = 14'd5215;	//
  parameter a8_const  = 13'd2608;	//
  parameter a9_const  = 12'd1304;	//
  parameter a10_const = 11'd652;	//
  parameter a11_const = 10'd326;	//
  parameter a12_const = 9'd163;		//
  parameter a13_const = 8'd81;		//
  parameter a14_const = 7'd41;		//
  parameter a15_const = 6'd20;		//
  parameter a16_const = 5'd10;		//
  parameter a17_const = 4'd5;		//

	// *********************
	// minimize rounding errors .. this also deletes the DC offset that the cordic was introducing
	
   always @ (negedge clk)
	begin
		i1b <= i1 + 21'd1;
		q1b <= q1 + 21'd1;

        i2b <= i2 + 21'd2;
        q2b <= q2 + 21'd2;

        i3b <= i3 + 21'd4;
        q3b <= q3 + 21'd4;

        i4b <= i4 + 21'd8;
        q4b <= q4 + 21'd8;

        i5b <= i5 + 21'd16;
        q5b <= q5 + 21'd16;

        i6b <= i6 + 21'd32;
        q6b <= q6 + 21'd32;

        i7b <= i7 + 21'd64;
        q7b <= q7 + 21'd64;

        i8b <= i8 + 21'd128;
        q8b <= q8 + 21'd128;

        i9b <= i9 + 21'd256;
        q9b <= q9 + 21'd256;

        i10b <= i10 + 21'd512;
        q10b <= q10 + 21'd512;

        i11b <= i11 + 21'd1024;
        q11b <= q11 + 21'd1024;

        i12b <= i12 + 21'd2048;
        q12b <= q12 + 21'd2048;

        i13b <= i13 + 21'd4096;
        q13b <= q13 + 21'd4096;

        i14b <= i14 + 21'd8192;
        q14b <= q14 + 21'd8192;

        i15b <= i15 + 21'd16384;
        q15b <= q15 + 21'd16384;

        i16b <= i16 + 21'd32768;
        q16b <= q16 + 21'd32768;

        i17b <= i17 + 21'd65536;
        q17b <= q17 + 21'd65536;

        i18b <= i18 + 21'd131072;
        q18b <= q18 + 21'd131072;

        i19b <= i19 + 21'd8;
        q19b <= q19 + 21'd8;
	end
      
	// *********************

	function [21:0] CordicIQ;
		input Add;   
		input [21:0] Val;
		input [21:0] Mod;
		begin
			CordicIQ = Add ? Val + Mod : Val - Mod;
		end
	endfunction

	always @ (posedge clk)
    begin
        // Prerotate: 90 degrees
        a0[20:0] <= {~ain[20], ain[19:0]};
        _in <= ain[21] ? -{in[15], in, 5'd0}: {in[15], in, 5'd0};
          
        // 45 degrees = atan(1.0 / 1)
        a1[19:0] <= {~a0[19], a0[18:0]};
        q1 <= _in;
        i1 <= a0[20] ? _in: -_in;
          
        // 26.56505118 degrees = atan(1.0 / 2)
        a2 <= a1[19] ? a1 + a1_const: a1 - a1_const;
        i2 <= CordicIQ( a1[19], i1, {q1b[21], q1b[21:1]});
        q2 <= CordicIQ(~a1[19], q1, {i1b[21], i1b[21:1]});

        // 14.03624347 degrees = atan(1.0 / 4)
        a3 <= a2[19] ? a2[18:0] + a2_const: a2[18:0] - a2_const;
        i3 <= CordicIQ( a2[19], i2, {{2{q2b[21]}}, q2b[21:2]});
        q3 <= CordicIQ(~a2[19], q2, {{2{i2b[21]}}, i2b[21:2]});

        // 7.125016349 degrees = atan(1.0 / 8)
        a4 <= a3[18] ? a3[17:0] + a3_const: a3[17:0] - a3_const;
        i4 <= CordicIQ( a3[18], i3, {{3{q3b[21]}}, q3b[21:3]});
        q4 <= CordicIQ(~a3[18], q3, {{3{i3b[21]}}, i3b[21:3]});

        // 3.576334375 degrees = atan(1.0 / 16)
        a5 <= a4[17] ? a4[16:0] + a4_const: a4[16:0] - a4_const;
        i5 <= CordicIQ( a4[17], i4, {{4{q4b[21]}}, q4b[21:4]});
        q5 <= CordicIQ(~a4[17], q4, {{4{i4b[21]}}, i4b[21:4]});

        // 1.789910608 degrees = atan(1.0 / 32)
        a6 <= a5[16] ? a5[15:0] + a5_const: a5[15:0] - a5_const;
        i6 <= CordicIQ( a5[16], i5, {{5{q5b[21]}}, q5b[21:5]});
        q6 <= CordicIQ(~a5[16], q5, {{5{i5b[21]}}, i5b[21:5]});

        // 0.8951737 degrees = atan(1.0 / 64)
        a7 <= a6[15] ? a6[14:0] + a6_const: a6[14:0] - a6_const;
        i7 <= CordicIQ( a6[15], i6, {{6{q6b[21]}}, q6b[21:6]});
        q7 <= CordicIQ(~a6[15], q6, {{6{i6b[21]}}, i6b[21:6]});

        // 0.4476141 degrees = atan(1.0 / 128)
        a8 <= a7[14] ? a7[13:0] + a7_const: a7[13:0] - a7_const;
        i8 <= CordicIQ( a7[14], i7, {{7{q7b[21]}}, q7b[21:7]});
        q8 <= CordicIQ(~a7[14], q7, {{7{i7b[21]}}, i7b[21:7]});

        // 0.2238105 degrees = atan(1.0 / 256)
        a9 <= a8[13] ? a8[12:0] + a8_const: a8[12:0] - a8_const;
        i9 <= CordicIQ( a8[13], i8, {{8{q8b[21]}}, q8b[21:8]});
        q9 <= CordicIQ(~a8[13], q8, {{8{i8b[21]}}, i8b[21:8]});

        // 0.1119056 degrees = atan(1.0 / 512)
        a10 <= a9[12] ? a9[11:0] + a9_const: a9[11:0] - a9_const;
        i10 <= CordicIQ( a9[12], i9, {{9{q9b[21]}}, q9b[21:9]});
        q10 <= CordicIQ(~a9[12], q9, {{9{i9b[21]}}, i9b[21:9]});

        // 0.0559528 degrees = atan(1.0 / 1024)
        a11 <= a10[11] ? a10[10:0] + a10_const: a10[10:0] - a10_const;
        i11 <= CordicIQ( a10[11], i10, {{10{q10b[21]}}, q10b[21:10]});
        q11 <= CordicIQ(~a10[11], q10, {{10{i10b[21]}}, i10b[21:10]});

        // 0.0279764 degrees = atan(1.0 / 2048)
        a12 <= a11[10] ? a11[9:0] + a11_const: a11[9:0] - a11_const;
        i12 <= CordicIQ( a11[10], i11, {{11{q11b[21]}}, q11b[21:11]});
        q12 <= CordicIQ(~a11[10], q11, {{11{i11b[21]}}, i11b[21:11]});

        // 0.013988227 degrees = atan(1.0 / 4096)
        a13 <= a12[9] ? a12[8:0] + a12_const: a12[8:0] - a12_const;
        i13 <= CordicIQ( a12[9], i12, {{12{q12b[21]}}, q12b[21:12]});
        q13 <= CordicIQ(~a12[9], q12, {{12{i12b[21]}}, i12b[21:12]});

        // 0.0069941137 degrees = atan(1.0 / 8192)
        a14 <= a13[8] ? a13[7:0] + a13_const: a13[7:0] - a13_const;
        i14 <= CordicIQ( a13[8], i13, {{13{q13b[21]}}, q13b[21:13]});
        q14 <= CordicIQ(~a13[8], q13, {{13{i13b[21]}}, i13b[21:13]});

        // 0.0034970569 degrees = atan(1.0 / 16384)
        a15 <= a14[7] ? a14[6:0] + a14_const: a14[6:0] - a14_const;
        i15 <= CordicIQ( a14[7], i14, {{14{q14b[21]}}, q14b[21:14]});
        q15 <= CordicIQ(~a14[7], q14, {{14{i14b[21]}}, i14b[21:14]});

        // 0.0017485284 degrees = atan(1.0 / 32768)
        a16 <= a15[6] ? a15[5:0] + a15_const: a15[5:0] - a15_const;
        i16 <= CordicIQ( a15[6], i15, {{15{q15b[21]}}, q15b[21:15]});
        q16 <= CordicIQ(~a15[6], q15, {{15{i15b[21]}}, i15b[21:15]});

        // 0.00087426421 degrees = atan(1.0 / 65536)
        a17 <= a16[5] ? a16[4:0] + a16_const: a16[4:0] - a16_const;
        i17 <= CordicIQ( a16[5], i16, {{16{q16b[21]}}, q16b[21:16]});
        q17 <= CordicIQ(~a16[5], q16, {{16{i16b[21]}}, i16b[21:16]});

        // 0.00043713211 degrees = atan(1.0 / 131072)
        a18 <= a17[4] ? a17[3:0] + a17_const: a17[3:0] - a17_const;
        i18 <= CordicIQ( a17[4], i17, {{17{q17b[21]}}, q17b[21:17]});
        q18 <= CordicIQ(~a17[4], q17, {{17{i17b[21]}}, i17b[21:17]});

        // 0.00021856605 degrees = atan(1.0 / 262144)
        i19 <= CordicIQ( a18[3], i18, {{18{q18b[21]}}, q18b[21:18]});
        q19 <= CordicIQ(~a18[3], q18, {{18{i18b[21]}}, i18b[21:18]});
		iout <= i19b[21:0];
		qout <= q19b[21:0];
    end

endmodule

